/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  useExchangeApiContext,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h } from "preact";
import { useOfficer } from "../hooks/officer.js";
import { HandleAccountNotReady } from "./HandleAccountNotReady.js";
import { useUiSettingsContext } from "../context/ui-settings.js";

export function Officer() {
  const officer = useOfficer();
  const settings = useUiSettingsContext();
  const { lib } = useExchangeApiContext();

  const { i18n } = useTranslationContext();
  if (officer.state !== "ready") {
    return <HandleAccountNotReady officer={officer} />;
  }

  const url = new URL("./", lib.exchange.baseUrl);
  const signupEmail = settings.signupEmail ?? `aml-signup@${url.hostname}`;

  return (
    <div>
      <h1 class="my-2 text-3xl font-bold tracking-tight text-gray-900 ">
        <i18n.Translate>Public key</i18n.Translate>
      </h1>
      <div class="max-w-xl text-base leading-7 text-gray-700 lg:max-w-lg">
        <p class="mt-6 font-mono break-all">{officer.account.id}</p>
      </div>
      <p>
        <a
          href={`mailto:${signupEmail}?subject=${encodeURIComponent(
            "Request AML signup",
          )}&body=${encodeURIComponent(
            `I want my AML account\n\n\nPubKey: ${officer.account.id}`,
          )}`}
          target="_blank"
          rel="noreferrer"
          class="m-4 block rounded-md w-fit border-0 px-3 py-2 text-center text-sm bg-indigo-700 text-white shadow-sm hover:bg-indigo-700"
        >
          <i18n.Translate>Request account activation</i18n.Translate>
        </a>
      </p>
      <p>
        <button
          type="button"
          onClick={() => {
            officer.lock();
          }}
          class="m-4 block rounded-md border-0 bg-gray-200 px-3 py-2 text-center text-sm text-black shadow-sm "
        >
          <i18n.Translate>Lock account</i18n.Translate>
        </button>
      </p>
      <p>
        <button
          type="button"
          onClick={() => {
            officer.forget();
          }}
          class="m-4 block rounded-md bg-red-600 px-3 py-2 text-center text-sm  text-white shadow-sm hover:bg-red-500 "
        >
          <i18n.Translate>Forget account</i18n.Translate>
        </button>
      </p>
    </div>
  );
}
