/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { TalerMerchantApi } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { h, VNode } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import { datetimeFormatForSettings, usePreference } from "../../../../hooks/preference.js";
import { WithId } from "../../../../declaration.js";

type Entity = TalerMerchantApi.TransferDetails & WithId;

interface Props {
  transfers: Entity[];
  onDelete: (id: Entity) => void;
  onCreate: () => void;
  accounts: string[];
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTable({
  transfers,
  onCreate,
  onDelete,
  onLoadMoreAfter,
  onLoadMoreBefore,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string[]>([]);

  const { i18n } = useTranslationContext();

  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-arrow-left-right" />
          </span>
          <i18n.Translate>Transfers</i18n.Translate>
        </p>
        <div class="card-header-icon" aria-label="more options">
          <span
            class="has-tooltip-left"
            data-tooltip={i18n.str`Add new transfer`}
          >
            <button class="button is-info" type="button" onClick={onCreate}>
              <span class="icon is-small">
                <i class="mdi mdi-plus mdi-36px" />
              </span>
            </button>
          </span>
        </div>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {transfers.length > 0 ? (
              <Table
                instances={transfers}
                onDelete={onDelete}
                rowSelection={rowSelection}
                rowSelectionHandler={rowSelectionHandler}
                onLoadMoreAfter={onLoadMoreAfter}
                onLoadMoreBefore={onLoadMoreBefore}
              />
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
interface TableProps {
  rowSelection: string[];
  instances: Entity[];
  onDelete: (id: Entity) => void;
  rowSelectionHandler: StateUpdater<string[]>;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

function Table({
  instances,
  onLoadMoreAfter,
  onDelete,
  onLoadMoreBefore,
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  const [settings] = usePreference();
  return (
    <div class="table-container">
      {onLoadMoreBefore && (
        <button
          class="button is-fullwidth"
          data-tooltip={i18n.str`Load more transfers before the first one`}
          onClick={onLoadMoreBefore}
        >
          <i18n.Translate>Load first page</i18n.Translate>
        </button>
      )}
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            <th>
              <i18n.Translate>ID</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Credit</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Confirmed</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Verified</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Executed at</i18n.Translate>
            </th>
            <th />
          </tr>
        </thead>
        <tbody>
          {instances.map((i) => {
            return (
              <tr key={i.id}>
                <td title={i.wtid}>{i.wtid.substring(0,16)}...</td>
                <td>{i.credit_amount}</td>
                <td>{i.confirmed ? i18n.str`yes` : i18n.str`no`}</td>
                <td>{i.verified ? i18n.str`yes` : i18n.str`no`}</td>
                <td>
                  {i.execution_time
                    ? i.execution_time.t_s == "never"
                      ? i18n.str`never`
                      : format(
                        i.execution_time.t_s * 1000,
                        datetimeFormatForSettings(settings),
                      )
                    : i18n.str`unknown`}
                </td>
                <td>
                  {i.verified !== true ? (
                    <button
                      class="button is-danger is-small has-tooltip-left"
                      data-tooltip={i18n.str`Delete selected transfer from the database`}
                      onClick={() => onDelete(i)}
                    >
                      <i18n.Translate>Delete</i18n.Translate>
                    </button>
                  ) : undefined}
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
      {onLoadMoreAfter && (
        <button
          class="button is-fullwidth"
          data-tooltip={i18n.str`Load more transfers after the last one`}
          onClick={onLoadMoreAfter}
        >
          <i18n.Translate>Load next page</i18n.Translate>
        </button>
      )}
    </div>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There is no transfer yet, add more pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}
