/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
  WalletNotification,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";
import { CoinConfig } from "../harness/denomStructures.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

const coinConfigList: CoinConfig[] = [
  {
    ...coinCommon,
    name: "n1",
    value: "TESTKUDOS:1",
  },
];

/**
 * Run a test for a multi-batch peer push payment.
 */
export async function runPeerPushLargeTest(t: GlobalTestState) {
  const { bank, exchange } = await createSimpleTestkudosEnvironmentV2(t, coinConfigList);

  let allW1Notifications: WalletNotification[] = [];
  let allW2Notifications: WalletNotification[] = [];

  const w1 = await createWalletDaemonWithClient(t, {
    name: "w1",
    handleNotification(wn) {
      allW1Notifications.push(wn);
    },
  });
  const w2 = await createWalletDaemonWithClient(t, {
    name: "w2",
    handleNotification(wn) {
      allW2Notifications.push(wn);
    },
  });

  // Withdraw digital cash into the wallet.

  const withdrawRes = await withdrawViaBankV2(t, {
    walletClient: w1.walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:300",
  });

  await withdrawRes.withdrawalFinishedCond;

  const purse_expiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );

  const checkResp0 = await w1.walletClient.call(
    WalletApiOperation.CheckPeerPushDebit,
    {
      amount: "TESTKUDOS:200" as AmountString,
    },
  );

  t.assertAmountEquals(checkResp0.amountEffective, "TESTKUDOS:200");

  const resp = await w1.walletClient.call(
    WalletApiOperation.InitiatePeerPushDebit,
    {
      partialContractTerms: {
        summary: "Hello World 🥺",
        amount: "TESTKUDOS:200" as AmountString,
        purse_expiration,
      },
    },
  );

  console.log(resp);

  const peerPushReadyCond = w1.walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.newTxState.major === TransactionMajorState.Pending &&
      x.newTxState.minor === TransactionMinorState.Ready &&
      x.transactionId === resp.transactionId,
  );

  await peerPushReadyCond;

  const txDetails = await w1.walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: resp.transactionId,
    },
  );
  t.assertDeepEqual(txDetails.type, TransactionType.PeerPushDebit);
  t.assertTrue(!!txDetails.talerUri);

  const checkResp = await w2.walletClient.call(
    WalletApiOperation.PreparePeerPushCredit,
    {
      talerUri: txDetails.talerUri,
    },
  );

  console.log(checkResp);

  const acceptResp = await w2.walletClient.call(
    WalletApiOperation.ConfirmPeerPushCredit,
    {
      transactionId: checkResp.transactionId,
    },
  );

  console.log(acceptResp);

  await w2.walletClient.call(
    WalletApiOperation.TestingWaitTransactionsFinal,
    {},
  );

  const txn1 = await w1.walletClient.call(
    WalletApiOperation.GetTransactions,
    {},
  );
  const txn2 = await w2.walletClient.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log(`txn1: ${j2s(txn1)}`);
  console.log(`txn2: ${j2s(txn2)}`);
}

runPeerPushLargeTest.suites = ["wallet"];
