/*
 Copyright 2019 Florian Dold

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 or implied. See the License for the specific language governing
 permissions and limitations under the License.
 */

import { BridgeIDBDatabaseInfo, BridgeIDBKeyRange } from "./bridge-idb.js";
import {
  IDBCursorDirection,
  IDBTransactionMode,
  IDBValidKey,
} from "./idbtypes.js";

export interface ConnectResult {
  conn: DatabaseConnection;
  version: number;
  objectStores: string[];
}

export interface DatabaseConnection {
  connectionCookie: string;
}

export interface DatabaseTransaction {
  transactionCookie: string;
}

export enum ResultLevel {
  OnlyCount,
  OnlyKeys,
  Full,
}

export enum StoreLevel {
  NoOverwrite,
  AllowOverwrite,
  UpdateExisting,
}


export interface IndexGetQuery {
  direction: IDBCursorDirection;
  objectStoreName: string;
  indexName: string;
  /**
   * The range of keys to return.
   * The range refers to the index keys.
   */
  range: BridgeIDBKeyRange | undefined | null;
  /**
   * Last cursor position in terms of the index key.
   * Can only be specified if lastObjectStorePosition is defined.
   *
   * Must either be undefined or within range.
   */
  lastIndexPosition?: IDBValidKey;
  /**
   * Last position in terms of the object store key.
   */
  lastObjectStorePosition?: IDBValidKey;
  /**
   * If specified, the index key of the results must be
   * greater or equal to advanceIndexKey.
   */
  advanceIndexKey?: IDBValidKey;
  /**
   * If specified, the primary key of the results must be greater
   * or equal to advancePrimaryKey.
   */
  advancePrimaryKey?: IDBValidKey;
  /**
   * Maximum number of results to return.
   * If 0, return all available results
   */
  limit: number;
  resultLevel: ResultLevel;
}

export interface ObjectStoreGetQuery {
  direction: IDBCursorDirection;
  objectStoreName: string;
  /**
   * The range of keys to return.
   * Refers to the object store keys.
   */
  range: BridgeIDBKeyRange | undefined | null;
  /**
   * Last position in terms of the object store key.
   */
  lastObjectStorePosition?: IDBValidKey;
  /**
   * If specified, the primary key of the results must be greater
   * or equal to advancePrimaryKey.
   */
  advancePrimaryKey?: IDBValidKey;
  /**
   * Maximum number of results to return.
   * If 0, return all available results
   */
  limit: number;
  resultLevel: ResultLevel;
}

export interface RecordGetResponse {
  values: any[] | undefined;
  indexKeys: IDBValidKey[] | undefined;
  primaryKeys: IDBValidKey[] | undefined;
  count: number;
}

export interface RecordStoreRequest {
  objectStoreName: string;
  value: any;
  key: IDBValidKey | undefined;
  storeLevel: StoreLevel;
}

export interface RecordStoreResponse {
  /**
   * Key that the record was stored under in the object store.
   */
  key: IDBValidKey;
}

export interface ObjectStoreMeta {
  indexSet: string[];
  keyPath: string | string[] | null;
  autoIncrement: boolean;
}

export interface IndexMeta {
  keyPath: string | string[];
  multiEntry: boolean;
  unique: boolean;
}

// FIXME: Instead of referring to an object store by name,
// maybe refer to it via some internal, numeric ID?
// This would simplify renaming.
export interface Backend {
  getDatabases(): Promise<BridgeIDBDatabaseInfo[]>;

  connectDatabase(name: string): Promise<ConnectResult>;

  beginTransaction(
    dbConn: DatabaseConnection,
    objectStores: string[],
    mode: IDBTransactionMode,
  ): Promise<DatabaseTransaction>;

  enterVersionChange(
    dbConn: DatabaseConnection,
    newVersion: number,
  ): Promise<DatabaseTransaction>;

  deleteDatabase(name: string): Promise<void>;

  close(dbConn: DatabaseConnection): Promise<void>;

  // FIXME: Use this for connection
  // prepareConnect() - acquires a lock, maybe enters a version change transaction?
  // finishConnect() - after possible versionchange is done, allow others to connect

  /**
   * Get metadata for an object store.
   *
   * When dbConn is running a version change transaction,
   * the current schema (and not the initial schema) is returned.
   *
   * Caller may mutate the result, a new object
   * is returned on each call.
   */
  getObjectStoreMeta(
    dbConn: DatabaseConnection,
    objectStoreName: string,
  ): ObjectStoreMeta | undefined;

  /**
   * Get metadata for an index.
   *
   * When dbConn is running a version change transaction,
   * the current schema (and not the initial schema) is returned.
   *
   * Caller may mutate the result, a new object
   * is returned on each call.
   */
  getIndexMeta(
    dbConn: DatabaseConnection,
    objectStoreName: string,
    indexName: string,
  ): IndexMeta | undefined;

  renameIndex(
    btx: DatabaseTransaction,
    objectStoreName: string,
    oldIndexName: string,
    newIndexName: string,
  ): void;

  deleteIndex(
    btx: DatabaseTransaction,
    objectStoreName: string,
    indexName: string,
  ): void;

  rollback(btx: DatabaseTransaction): void;

  // FIXME: Should probably not be async
  commit(btx: DatabaseTransaction): Promise<void>;

  deleteObjectStore(btx: DatabaseTransaction, name: string): void;

  createObjectStore(
    btx: DatabaseTransaction,
    name: string,
    keyPath: string | string[] | null,
    autoIncrement: boolean,
  ): void;

  renameObjectStore(
    btx: DatabaseTransaction,
    oldName: string,
    newName: string,
  ): void;

  createIndex(
    btx: DatabaseTransaction,
    indexName: string,
    objectStoreName: string,
    keyPath: string | string[],
    multiEntry: boolean,
    unique: boolean,
  ): void;

  deleteRecord(
    btx: DatabaseTransaction,
    objectStoreName: string,
    range: BridgeIDBKeyRange,
  ): Promise<void>;

  getObjectStoreRecords(
    btx: DatabaseTransaction,
    req: ObjectStoreGetQuery,
  ): Promise<RecordGetResponse>;

  getIndexRecords(
    btx: DatabaseTransaction,
    req: IndexGetQuery,
  ): Promise<RecordGetResponse>;

  storeRecord(
    btx: DatabaseTransaction,
    storeReq: RecordStoreRequest,
  ): Promise<RecordStoreResponse>;

  clearObjectStore(
    btx: DatabaseTransaction,
    objectStoreName: string,
  ): Promise<void>;
}
