import re, time, os, json
from difflib import SequenceMatcher
from resources.lib.parsedom import parseDOM
from resources.lib.bs4 import BeautifulSoup
import utils, log_utils, common


EPG_SOURCES_CACHE_FILE = os.path.join(common.cache_path, 'epg_sources')
EPG_SOURCES_CACHE_EXPIRE = 5 * 24 * 60 * 60


def saveEpgSourcesToCache(data):
    try:
        with open(EPG_SOURCES_CACHE_FILE, 'w') as f:
            f.write(json.dumps(data))
    except:
        pass


def loadEpgSourcesFromCache():
    data = []
    try:
        if os.path.isfile(EPG_SOURCES_CACHE_FILE):
            if (os.path.getsize(EPG_SOURCES_CACHE_FILE) > 0) and ((os.path.getmtime(EPG_SOURCES_CACHE_FILE) - time.time()) <= EPG_SOURCES_CACHE_EXPIRE):
                with open(EPG_SOURCES_CACHE_FILE, 'r') as f:
                    data = json.loads(f.read())
            else:
                os.remove(EPG_SOURCES_CACHE_FILE)
    except:
        pass
    
    return data


def getChannelGuideUrl(channel_name):
    sources = loadEpgSourcesFromCache()
    
    if not sources:
        url = 'https://www.cinemagia.ro/program-tv/'
        
        html = utils.get_html(url)

        if not html:
            return ''

        names = parseDOM(html, 'a', attrs={'class': 'station-link'})    
        urls = parseDOM(html, 'a', attrs={'class': 'station-link'}, ret='href')
        
        if len(names) != len(urls):
            return ''
        
        for i in xrange(len(names)):
            sources.append({
                'channel_name': utils.unescape(names[i], True),
                'url': urls[i]
            })
        
        saveEpgSourcesToCache(sources)
    
    seq2 = channel_name.lower().replace(' ', '').strip()
    
    seqm = SequenceMatcher()
    seqm.set_seq2(seq2)
    
    index_with_max_ratio = -1
    
    for i, source in enumerate(sources):
        seq1 = source['channel_name'].lower().replace(' ', '').strip()
        seqm.set_seq1(seq1)
        ratio = seqm.ratio()

        if ratio >= 0.8 and i > index_with_max_ratio:
            index_with_max_ratio = i
        
        if ratio == 1.0:
            break
    
    if index_with_max_ratio == -1:
        return ''
    
    return sources[index_with_max_ratio]['url']


def getTVGuide(channel_name):
    url = getChannelGuideUrl(channel_name)
    
    html = utils.get_html(url)
    
    if not html:
        return []
    
    soup = BeautifulSoup(html, 'html5lib')
    tds = soup.findAll('td', attrs={'class': 'container_events'})
    tds = [tds[i] for i in xrange(len(tds)) if divmod(i, 4)[1] == 0]
    
    hours = []
    titles = []
    
    for td in tds:
        hours.extend(td.findAll('td', attrs={'class': 'ora'}))
        titles.extend(td.findAll('div', attrs={'class': 'title'}))
    
    if not hours or not titles or len(hours) != len(titles):
        return []
    
    programs = []
    
    for i in xrange(len(titles)):
        current = 'current' in str(hours[i])
        hour = re.search(r'<div>(\d+:\d+)<\/div>', str(hours[i])).group(1)
        title = titles[i].getText().strip()
        title = ' '.join(title.split())
        title = utils.unescape(title, True)
        item = (hour, title, current)
        programs.append(item)
    
    return programs
